function [] = Newton_NLPDE(N,gridtype,s,beta,rho)
% Newton_NLPDE
% Script that performs Kansa collocation within Newton iteration for 2D nonlinear PDE
% N: Number of (interior) collocation points, should be a square (e.g. 25,
% 81, 289, 1089)
% gridtype: type of collocation points
%  'u'=uniform, 'h'=Halton, 'c'=Chebyshev
% s,beta: parameters for Matern functions
% rho: basis for smoothing parameter, i.e., smoothing = rho^(-theta*b^k),
%      where theta=1.1435 and b=1.2446 below, k iteration
%      Smoothing is accomplished via convolution 
%            phi(s,beta) * phi(s,smoothing) = phi(s,beta+smoothing)
% Calls on: rbf_definitionMatern (special version for Matern functions), DistanceMatrix

%close all

% Fontsize for plots: 14 for papers, 16 for presentation
fsize = 16;

global rbf Lrbf
rbf_definitionMatern;
ep = sqrt(N)/2; % Shape parameter for RBF
mu = .1; % boundary layer
% Number of levels for Newton iteration
maxlevel = 50;

  % Exact solution and its Laplacian for test problem
  expc = 1 + exp(-1/mu);
  exps = @(x,mu) exp(-x/mu) + exp((x-1)/mu);
  psi = @(x,mu) expc - exps(x,mu);
  u = @(x,y,mu) psi(x,mu).*psi(y,mu);
  Lu = @(x,y,mu) -(exps(x,mu)-2*exp(-(x+y)/mu)-2*exp(-(x-y+1)/mu)+exp((x-1)/mu)+exp((x-2)/mu)-2*exp((x-1-y)/mu)-2*exp((x-2+y)/mu)+exps(y,mu)+exp(-(1+y)/mu)+exp((-2+y)/mu))/mu^2;
  % right-hand side function
  f = @(x,y,mu) exps(x,mu).*psi(y,mu) + exps(y,mu).*psi(x,mu) - psi(x,mu).*psi(y,mu) + (psi(x,mu).*psi(y,mu)).^3;

  neval = 40;
  % Compute (interior) collocation points
   switch gridtype
       case 'u'
           [x1, x2] = ndgrid(linspace(0,1,sqrt(N)));
           dsites = [x1(:) x2(:)];
           fname = 'Uniform';
       case 'h'
           dsites = haltonseq(N,2);
           [x1, x2] = ndgrid(linspace(0,1,sqrt(N)));   % for plotting of initial guess only
           fname = 'Halton';
       case 'c'
           [x,y] = meshgrid(0.5+0.5*cos(pi*(0:sqrt(N)-1)/(sqrt(N)-1)));
           dsites = [reshape(x,N,1),reshape(y,N,1)];
           x1 = x; x2 = y;
           fname = 'Chebyshev';
       otherwise
           error('Please use u, h, or c data types')
   end          
  intdata = dsites;

  % Additional (equally spaced) boundary collocation points
  sn = sqrt(N); bdylin = linspace(0,1,sn)';
  bdy0 = zeros(sn-1,1); bdy1 = ones(sn-1,1);
  bdydata = [bdylin(1:end-1) bdy0; bdy1 bdylin(1:end-1);...
       flipud(bdylin(2:end)) bdy1; bdy0 flipud(bdylin(2:end))];
  % Create additional boundary centers OUTSIDE the domain
  h = 1/(sn-1); bdylin = (h:h:1-h)';
  bdy0 = -h*ones(sn-2,1); bdy1 = (1+h)*ones(sn-2,1);
  bdyctrs = [-h -h; bdylin bdy0; 1+h -h; bdy1 bdylin;...
      1+h 1+h; flipud(bdylin) bdy1; -h 1+h; bdy0 flipud(bdylin)];
  % If outside centers not used (OK for interior Halton pts)
  % bdyctrs=bdydata;
  ctrs = [intdata; bdyctrs];

  % Plot collocation points and centers
  figure
  hold on; plot(intdata(:,1),intdata(:,2),'bo','Linewidth',2);
  plot(bdydata(:,1),bdydata(:,2),'rx','Linewidth',2);
  plot(bdyctrs(:,1),bdyctrs(:,2),'gx','Linewidth',2); hold off
  xlim([-0.2 1.2])
  set(gca,'Fontsize',fsize,'Linewidth',2)
  print('-dpng',[fname int2str(N) '.png'])
  
  % Initial guess (take bi-quadratic)
  u_old = 16*intdata(:,1).*(1-intdata(:,1)).*intdata(:,2).*(1-intdata(:,2));
  Lu_old = 64;
  % Create initial right-hand side
  rhs = [f(intdata(:,1),intdata(:,2),mu) - (-mu^2*Lu_old-u_old+u_old.^3); ...
         zeros(size(bdydata(:,1)))];

  % Plot initial guess
  figure
  caption = sprintf('Initial guess');
  Pfplot = surf(x1,x2,reshape(u_old,sqrt(N),sqrt(N)),reshape(u_old,sqrt(N),sqrt(N)));
  set(Pfplot,'FaceColor','interp','EdgeColor','none')
  set(gca,'Fontsize',fsize,'Linewidth',2)
  xlabel('x','FontSize',fsize);
  ylabel('y','FontSize',fsize);
  zlabel('z','FontSize',fsize,'Rotation',0);
  view([-30,30]); axis([0 1 0 1 -0.2 1.2]) 
  colormap hsv
  vcb = colorbar('vert');
  set(get(vcb,'YLabel'),'String','Function Value','FontSize',fsize)
 % title(caption)
  print('-dpng','InitialGuess.png')

  % Create neval-by-neval equally spaced evaluation locations
  % in the unit square
  grid = linspace(0,1,neval); [xe,ye] = meshgrid(grid);
  epoints = [xe(:) ye(:)];
    
  % Exact solution for error computation and distance matrix for evaluation
  exact = u(epoints(:,1),epoints(:,2),mu);
  DM_eval = DistanceMatrix(epoints,ctrs);
  
  % Compute distance matrices for blocks of collocation matrix
  DM_intdata = DistanceMatrix(intdata,ctrs);
  DM_bdydata = DistanceMatrix(bdydata,ctrs);

  Pf_old = zeros(neval^2,1);
  Rf_Lu_old = zeros(size(intdata(:,1)));
  Rf_int_old = zeros(size(intdata(:,1)));
  Rf_bdy_old = zeros(size(bdydata(:,1)));

smoothinglist = [];
resnorm = [];

  for level=1:maxlevel
     disp(sprintf('LEVEL %d', level))
  
     LF = repmat(3*u_old.^2-1,1,size(ctrs,1));
     
     % Compute blocks for collocation matrix
     LCMLu = Lrbf(ep,DM_intdata,s,beta);
     LCMu = rbf(ep,DM_intdata,s,beta);
     BCM = rbf(ep,DM_bdydata,s,beta);

     % Update collocation matrix
     CM = [-mu^2*LCMLu + LF.*LCMu; ...
           BCM];
     if level == 1 
         size(CM)
     end
     clear LF
     
     if rho>0   % Nash smoothing with convolved Matern
         clear LCMLu LCMu BCM
        b = 1.2446;   % Jerome
        theta = 1.1435;
        smoothing = rho^(theta*b^level);
        smoothinglist = [smoothinglist smoothing];
        disp(sprintf('Smoothing with %f', smoothing))
        % Compute the matrices for smoothed evaluation
        EM = rbf(ep,DM_eval,s,beta+smoothing);
        LEMLu = Lrbf(ep,DM_intdata,s,beta+smoothing);
        LEMu = rbf(ep,DM_intdata,s,beta+smoothing);
        BEM = rbf(ep,DM_bdydata,s,beta+smoothing);
     else
         % Compute matrices for evaluation without smoothing
         EM = rbf(ep,DM_eval,s,beta);
         LEMLu = LCMLu;
         LEMu = LCMu;
         BEM = BCM;
         clear LCMLu LCMu BCM
     end

     % Update RBF solution
     coeff = CM\rhs;
     Pf = Pf_old + EM*coeff;
     
     % Solution on computational grid (for residuals)
     Rf_Lu = Rf_Lu_old + LEMLu*coeff;
     Rf_int = Rf_int_old + LEMu*coeff;
     Rf_bdy = Rf_bdy_old + BEM*coeff;
     
     % Create new right-hand side
     rhs = [f(intdata(:,1),intdata(:,2),mu) - (-mu^2*Rf_Lu-Rf_int+Rf_int.^3);  ...
            zeros(size(bdydata(:,1)))- Rf_bdy];

     resnorm = [resnorm norm(rhs)];
     
     Pf_old = Pf;
     u_old = Rf_int;
     Rf_int_old = Rf_int;
     Rf_bdy_old = Rf_bdy;
     Rf_Lu_old = Rf_Lu;
     
     % Compute maximum error on evaluation grid
     maxerr(level) = max(abs(Pf - exact));
     rms_err(level) = norm(Pf-exact)/neval;
     disp(sprintf('RMS error:     %e', rms_err(level)))
     disp(sprintf('Maximum error: %e', maxerr(level)))
     if (level > 1)
        max_rate = log(maxerr_old/maxerr(level))/log(2);
        rms_rate = log(rms_err_old/rms_err(level))/log(2);
        disp(sprintf('RMS rate:      %f', rms_rate))
        disp(sprintf('Maxerror rate: %f', max_rate))
     end
     if (level > 1 && max_rate < 0.001)
         break
     end
     maxerr_old = maxerr(level);
     rms_err_old = rms_err(level);

     clear EM LEMLu LEMu BEM CM
          
     % Plot update surface
%     figure
%     Pfplot = surf(xe,ye,reshape(EM*coeff,neval,neval),reshape(abs(EM*coeff),neval,neval));
%     set(Pfplot,'FaceColor','interp','EdgeColor','none')
%  set(gca,'Fontsize',14)
%  xlabel('x','FontSize',14);
%  ylabel('y','FontSize',14);
%  zlabel('z','FontSize',14,'Rotation',0);
%%     [cmin cmax] = caxis;
%%     caxis([cmin-.25*maxerr(level) cmax]);
%     view([-30,30]); 
%     colormap hsv
%%     vcb = colorbar('vert');
%%     ylim(vcb,[0 maxerr(level)])
%%     title('Nonsymmetric RBF update false colored by abs function value.')
%%pause
     % Plot solution surface
%     figure
%     Pfplot = surf(xe,ye,reshape(Pf,neval,neval),reshape(abs(Pf-exact),neval,neval));
%     set(Pfplot,'FaceColor','interp','EdgeColor','none')
%  set(gca,'Fontsize',14)
%  xlabel('x','FontSize',14);
%  ylabel('y','FontSize',14);
%  zlabel('z','FontSize',14,'Rotation',0);
%%     [cmin cmax] = caxis;
%%     caxis([cmin-.25*maxerr(level) cmax]);
%     view([-30,30]); 
%     colormap hsv
%%     vcb = colorbar('vert');
%%     ylim(vcb,[0 maxerr(level)])
%%     title('Nonsymmetric RBF solution false colored by maximum error.')
%%pause
  end

  figure
  semilogy(1:length(rms_err),rms_err,'r','Linewidth',2);
  set(gca,'Fontsize',14)
  xlabel('k','FontSize',14);
  ylabel('Errors','FontSize',14);

  figure
  semilogy(1:length(resnorm),resnorm,'b','Linewidth',2);
  set(gca,'Fontsize',14)
  xlabel('k','FontSize',14);
  ylabel('Residual norm','FontSize',14);

  if rho > 0
      hold on
      semilogy(1:length(smoothinglist),smoothinglist,'r','Linewidth',2);
      set(gca,'Fontsize',14)
      xlabel('k','FontSize',14);
      ylabel('Residual norm, \alpha','FontSize',14);
      legend('Residual norm', '\alpha')
      hold off
  end

  % Plot error surface
  figure
  errorplot = surf(xe,ye,reshape(abs(Pf-exact),neval,neval));
  set(errorplot,'FaceColor','interp','EdgeColor','none')
  set(gca,'Fontsize',14)
  xlabel('x','FontSize',14);
  ylabel('y','FontSize',14);
  zlabel('Error','FontSize',14);
  view([-30,30]);
  colormap hsv

  % Plot true solution
  figure
  caption = sprintf('Exact solution');
  Pfplot = surf(xe,ye,reshape(exact,neval,neval),reshape(exact,neval,neval));
  set(Pfplot,'FaceColor','interp','EdgeColor','none')
  set(gca,'Fontsize',fsize,'Linewidth',2)
  xlabel('x','FontSize',fsize);
  ylabel('y','FontSize',fsize);
  zlabel('z','FontSize',fsize,'Rotation',0);
  view([-30,30]); axis([0 1 0 1 -0.2 1.2]) 
  colormap hsv
  vcb = colorbar('vert');
  set(get(vcb,'YLabel'),'String','Function Value','FontSize',fsize)
%  title(caption)
  print('-dpng','ExactSolution.png')


