function N = KansaLaplaceMixedBCTref_2D(N, epsilon)
% function N = KansaLaplaceMixedBCTref_2D(N, epsilon)
% For example, use N=289, epsilon=3 with Gaussians
% Script that performs basic Kansa collocation for 2D Laplace equation (Prog 36 in Trefethen)
%
% Calls on: rbf_definition, DistanceMatrix
%
% no error checking

%%%%%%%%%%%%%%%%%% Editable Section %%%%%%%%%%%%%%%%%%%
% 
% The RBF is defined in the function rbf_definition 
% (separate file rbf_definition has to be edited to get desired RBF, 
% however, this file can be accessed by all programs)
global rbf Lrbf;  rbf_definition; 
% Test problem from Trefethen Prog.36
% Domain [-1,1]^2
u = @(x,y) zeros(size(x));
Lu = @(x,y) zeros(size(x));

% Number of evaluation points for plotting
M = 1681;      % to create neval-by-neval evaluation grid in unit square
%
%%%%%%%%%%%%%%%%%% Code below this line need not be changed %%%%%%%%%%%%%%%%
[collpts, N] = CreatePoints(N, 2, 'u');
collpts = 2*collpts-1;
indx = find(abs(collpts(:,1))==1 | abs(collpts(:,2))==1);
bdypts = collpts(indx,:); %%% finding the points on the boundary 
intpts = collpts(setdiff([1:N],indx),:); %% finds interior points

% Let centers coincide with ALL data sites (can be done for Halton points)
ctrs = [intpts; bdypts];

% Create neval-by-neval equally spaced evaluation locations in the unit square
evalpts = CreatePoints(M,2,'u');
evalpts = 2*evalpts-1;

% Compute the distance matrix between the evaluation points and centers
DM_eval = DistanceMatrix(evalpts,ctrs);
% Compute evaluation matrix
EM = rbf(epsilon,DM_eval);

% Now we find the RBF interpolant (this should not require any changes)
% Compute the distance matrix between the data sites and centers
DM_int = DistanceMatrix(intpts,ctrs);
DM_bdy = DistanceMatrix(bdypts,ctrs);

% Compute blocks for interpolation matrix
LCM = Lrbf(epsilon,DM_int);
BCM = rbf(epsilon,DM_bdy);

% non-symmetric interpolation matrix
CM = [LCM; BCM];
    
% Create right-hand side vector, i.e., evaluate the test function at the data points.
% If rhs read from data file, then comment this line out
rhs = zeros(N,1);  NI = size(intpts,1);
indx = find(bdypts(:,1)==1 | (bdypts(:,1)<0) & (bdypts(:,2)==1));
rhs(NI+indx) = (bdypts(indx,1)==1)*0.2.*sin(3*pi*bdypts(indx,2)) + ...
    (bdypts(indx,1)<0).*(bdypts(indx,2)==1).*sin(pi*bdypts(indx,1)).^4;

% Compute RBF interpolant (evaluation matrix * solution of interpolation system)
Pf = EM * (CM\rhs);
disp(sprintf('u(0,0) = %16.12f',Pf(841)))

% Plot data sites
hold on
plot(intpts(:,1),intpts(:,2),'bo');
plot(bdypts(:,1),bdypts(:,2),'rx');
caption = sprintf('Data sites for %d points', N);
title(caption)
hold off

% Plot centers sites
figure
hold on
plot(ctrs(:,1),ctrs(:,2),'bo');
caption = sprintf('Centers for %d points', N);
title(caption)
hold off

% Plot interpolant
figure
fview = [-20,45];
xe = reshape(evalpts(:,1),sqrt(M),sqrt(M));
ye = reshape(evalpts(:,2),sqrt(M),sqrt(M));
Pfplot = surf(xe,ye,reshape(Pf,sqrt(M),sqrt(M)));
view(fview);
axis([-1 1 -1 1 -.2 1])
text(0,.8,.5,sprintf('u(0,0) = %12.10f',Pf(841)))

