function [maxerr, rms_err, N] = IAMLSPrecond_sD(N, s, gridtype, D, n)
% function [maxerr, rms_err] = IAMLSPrecond_sD(N, s, gridtype, D, n)
% Uses accelerated iterated AMLS approximation to precondition sD RBF 
% interpolation system (compares backslash solution, pinv, GMRES, PCG)
% Inputs:
% N: number of interpolation points
% s: space dimension
% gridtype: Type of data points: 'c'=Chebyshev, 'f'=fence(rank-1 lattice),
% 'h'=Halton, 'l'=latin hypercube, 'n'=random normal, 'r'=random uniform,
% 's'=Sobol, 'u'=uniform   
% D: Mazya/Schmidt shape parameter
% n: number of IAMLS iterations
% Outputs:
% maxerr: maximum error on set of M (set below) random points 
% rms_err: RMS error on set of M (set below) random points 
% N: actual number of data points used (could be smaller for 'c' and 'u'
% types)
% Calls on: rbf_definition, testfunctionsD, DistanceMatrixB

%%%%%%%%%%%%%%%%%% Editable Section %%%%%%%%%%%%%%%%%%%%

% Number of evaluation points for error computation
M = 500;      % to create evaluation grid in unit cube

% The RBF is defined in the function rbf_definition 
% (separate file rbf_definition has to be edited to get desired RBF, 
% however, this file can be accessed by all programs)
global rbf
rbf_definition;

%%%%%%%%%%%%%%%%%% Code below this line need not be changed %%%%%%%%%%%%%%%%

[datasites, N] = CreatePoints(N,s,gridtype);
% Let centers coincide with data sites
centers = datasites;
rhs = testfunctionsD(datasites);

h = 1/(nthroot(N,s)-1);
ep = 1/(sqrt(D)*h)

% Create evaluation locations in the unit cube
[evalpoints, M] = CreatePoints(M,s,'r');
exact = testfunctionsD(evalpoints);

% Evaluate the test function at the data points
rhs = testfunctionsD(datasites);

% Now we find the RBF interpolant (this should not require any changes)
% Compute the distance matrix between the data sites and centers
DM_data = DistanceMatrixB(datasites,centers);
IM = rbf(ep,DM_data)/(sqrt(pi*D)^s);
condA = cond(IM);

% Compute the distance matrix between the evaluation points and centers
DM_eval = DistanceMatrixB(evalpoints,centers);
EM = rbf(ep,DM_eval)/(sqrt(pi*D)^s);

% right preconditioned IAMLS solution (APC=b) with acceleration, GMRES
% solution
disp('Preconditioner with acceleration')
P = eye(N);
AP = IM*P;
for i=1:n
    P = P*(2*eye(N)-AP);
    AP = IM*P;
end

[c FLAG] = gmres(AP, rhs, [], 1e-10, 40);  
Pf = (EM*P)*c;

maxerr = norm(Pf-exact,inf);
rms_err = norm(Pf-exact)/sqrt(M);
fprintf('RMS error:     %e\n', rms_err)
fprintf('Maximum error: %e\n', maxerr)

