% function N = HermiteLaplaceMixedBCTref_2D(N, epsilon)
%
% Function that performs basic Hermite collocation for 2D Laplace equation (Prog 36 in Trefethen)
% Note: exact solution not known
%
% Calls on: rbf_definition, DistanceMatrix, plotsurf, ploterror2D
%
% Example: Gaussians, epsilon=6, N=289
% no error checking

function N = HermiteLaplaceMixedBCTref_2D(N, epsilon)

%%%%%%%%%%%%%%%%%% Editable Section %%%%%%%%%%%%%%%%%%%%
% 
% The RBF is defined in the function rbf_definition 
% (separate file rbf_definition has to be edited to get desired RBF, 
% however, this file can be accessed by all programs)
global rbf Lrbf L2rbf;  rbf_definition;
% Test problem from Trefethen Prog.36
% Domain [-1,1]^2
u = @(x,y) zeros(size(x));
Lu = @(x,y) zeros(size(x));

% Number of evaluation points for plotting
M = 1681;      % to create neval-by-neval evaluation grid in unit square
%
%%%%%%%%%%%%%%%%%% Code below this line need not be changed %%%%%%%%%%%%%%%%
[datasites, N] = CreatePoints(N, 2, 'u');
intdata = 2*datasites-1;
% Additional boundary points
sg = sqrt(N); bdylin = linspace(-1,1,sg)'; bdy1 = ones(sg-1,1);
bdydata = [bdylin(1:end-1) -bdy1; bdy1 bdylin(1:end-1);...
           flipud(bdylin(2:end)) bdy1; -bdy1 flipud(bdylin(2:end))];

% Let centers coincide with ALL data sites (can be done for Halton points)
%bdycenters = bdydata;
%centers = [intdata; bdydata];

% Create additional boundary centers OUTSIDE the domain
h = 2/(sg-1); bdylin = (-1+h:h:1-h)'; bdy0 = repmat(-1-h,sg-2,1); bdy1 = repmat(1+h,sg-2,1);
bdycenters = [-1-h -1-h; bdylin bdy0; 1+h -1-h; bdy1 bdylin;...
              1+h 1+h; flipud(bdylin) bdy1; -1-h 1+h; bdy0 flipud(bdylin)]; 
centers = [intdata; bdycenters];
% Create neval-by-neval equally spaced evaluation locations in the unit square
evalpoints = CreatePoints(M, 2, 'u');
evalpoints = 2*evalpoints-1;

% Compute the distance matrix between the evaluation points and centers
DM_inteval = DistanceMatrix(evalpoints,intdata);
DM_bdyeval = DistanceMatrix(evalpoints,bdycenters);

% Compute evaluation matrix
LEM = Lrbf(epsilon,DM_inteval);
BEM = rbf(epsilon,DM_bdyeval);

EM = [LEM BEM];

% Now we find the RBF interpolant (this should not require any changes)
% Compute the distance matrix between the data sites and centers
DM_IIdata = DistanceMatrix(intdata,intdata);
DM_IBdata = DistanceMatrix(intdata,bdycenters);
DM_BIdata = DistanceMatrix(bdydata,intdata);
DM_BBdata = DistanceMatrix(bdydata,bdycenters);

% Compute blocks for interpolation matrix 
LLCM = L2rbf(epsilon,DM_IIdata);
LBCM = Lrbf(epsilon,DM_IBdata);
BLCM = Lrbf(epsilon,DM_BIdata);
BBCM = rbf(epsilon,DM_BBdata);

% symmetric interpolation matrix
CM = [LLCM LBCM; BLCM BBCM];
    
% Create right-hand side vector, i.e., evaluate the test function at the data points.
% If rhs read from data file, then comment this line out
rhs = [Lu(intdata(:,1),intdata(:,2)); zeros(sg-1,1); 0.2*sin(3*pi*bdydata(sg:2*sg-2,2)); ...
        zeros((sg-1)/2,1); sin(pi*bdydata((5*sg-3)/2:3*sg-3,1)).^4; zeros(sg-1,1)];

% Compute RBF interpolant (evaluation matrix * solution of interpolation system)
Pf = EM * (CM\rhs);
disp(sprintf('u(0,0) = %16.12f',Pf(841)))

% Plot data sites
hold on
plot(intdata(:,1),intdata(:,2),'bo');
plot(bdydata(:,1),bdydata(:,2),'rx');
caption = sprintf('Data sites for %d interior and %d boundary points', N,4*(sqrt(N)-1));
title(caption)
hold off

% Plot centers sites
figure
hold on
plot(intdata(:,1),intdata(:,2),'bo');
plot(bdycenters(:,1),bdycenters(:,2),'rx');
caption = sprintf('Centers for %d interior and %d boundary points', N,4*(sqrt(N)-1));
title(caption)
hold off

% Plot interpolant
figure
fview = [-20,45];
xe = reshape(evalpoints(:,1),sqrt(M),sqrt(M));
ye = reshape(evalpoints(:,2),sqrt(M),sqrt(M));
Pfplot = surf(xe,ye,reshape(Pf,sqrt(M),sqrt(M)));
view(fview);
axis([-1 1 -1 1 -.2 1])
text(0,.8,.5,sprintf('u(0,0) = %12.10f',Pf(841)))

