function [points, N] = CreatePoints(N,s,gridtype)
% Computes a set of N points in [0,1]^s
% Note: could add variable interval later
% Inputs:
% N: number of interpolation points
% s: space dimension
% gridtype: 'c'=Chebyshev, 'f'=fence(rank-1 lattice),
%    'h'=Halton, 'l'=latin hypercube, 'n'=random normal, 
%    'r'=random uniform, 's'=Sobol, 'u'=uniform   
% Outputs:
% points: an Nxs matrix (each row contains one s-D point)
% N: might be slightly less than original N for 
%    Chebyshev and gridded uniform points
% Calls on: chebsamp, lattice, haltonseq, lhsamp, i4_sobol, 
%           gridsamp
% Also needs: fdnodes,gaussj,i4_bit_hi1,i4_bit_lo0,i4_xor

switch gridtype
    case 'c'
        ppd = zeros(1,s);
        for j=1:s
            ppd(j) = floor(nthroot(N,s+1-j));
            N = N/ppd(j);
        end
        gam = 0.5*ones(1,s);  % density for point distribution, 0.5=Chebyshev
        points = chebsamp([zeros(1,s); ones(1,s)], ppd, gam);
        N = prod(ppd);
    case 'f'
        points = lattice(N,s);  % N should be(?) power of 2
    case 'h' 
        points = haltonseq(N,s);
    case 'l'
        points = lhsamp(N,s);
    case 'n'
        randn('state',47); 
        points = 0.5 + 0.1*randn(N,s);
    case 'r'
        rand('state',47); 
        points = rand(N,s);
    case 's'
        sp = max(2,s);
        points = zeros(N,sp);
        seed = 0; for i=1:N; [points(i,:) seed] = i4_sobol(sp,seed); end
        points = points(:,1:s);
        points = N*points/(N-1);
    case 'u'
        ppd = zeros(1,s);
        for j=1:s
            ppd(j) = floor(nthroot(N,s+1-j));
            N = N/ppd(j);
        end
        points = gridsamp([zeros(1,s); ones(1,s)], ppd);
        N = prod(ppd);
    otherwise
        error('Please use c, f, h, n, r, s or u data types')
end
